using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Drawing.Imaging;
using System.Text;
using System.Windows.Forms;
using System.IO.Ports;
using IdParserM280Works.Properties;//needed to save app.config
using BuffMgr;//needed to build a scan or swipe
using FileManager;
using System.Threading;
using PdfQLibNet; //referenced from folder named DLLs
using MagQLibNet; //referenced from folder named DLLs
using CyUSB; //M280 image v 1.2.3.0, referenced from folder named DLLs
using System.Media; //playbeep
using Microsoft.Win32; //for power event
//Note: The file QLMLicenseLib.dll must be copied to the deploy directory, to accomplish this, we make a reference to the file found in the DLLs folder.

/*****************************************************************/
/* Tokenworks PDF417 and Magnetic Stripe Parsing Library        */
/*  Copyright 2023, TokenWorks Inc.                      */
/* All rights reserved.                                          */
/* Redistribution and use of this code in source and/or binary   */
/* forms, with or without modification, are permitted provided   */
/* that: (1) all copies of the source code retain the above      */
/* unmodified copyright notice and this entire unmodified        */
/* section of text, (2) You or Your organization owns a valid    */
/* Developer License to this product from TokenWorks Inc.       */
/* and, (3) when any portion of this code is bundled in any      */
/* form with an application, a valid notice must be provided     */
/* within the user documentation, start-up screen or in the      */
/* help-about section of the application that specifies          */
/* TokenWorks as the provider of the Software bundled      */
/* with the application.                                         */
/*****************************************************************/


// SDK Version 2.8 - Inital Release of M280 card parsing and image capture
// SDK Version 2.9.1 - parsing updates
// SDK Version 3.0 - parsing updates
// SDK Version 3.1 - parsing updates 
// SDK Version 3.2 - parsing updates, new Crypto field
// SDK Version 3.2.1 - Add ScannerBusy bool to image capture, recompile in VS2015 to improve CyUSB reliablity
// SDK Version 3.3 - skipped
// SDK Version 3.4 - Remove Crypto field
// SDK Version 3.5 - Add Country field parsing to PDF
// SDK Version 3.6 - update PDF to rev 3.6 to fix old IL cards with HID scanners, update M280 image capture code to fix Win10 May 2018 update
// SDK Version 3.7 - update Mag for new MN magstripe format, update PDF to add Gender X to AAMVA barcodes
// SDK Version 3.8 - update Mag for BC city name > 13 char, update PDF for BC city name >13, OR 2nd Addr, NJ title field parsing
// SDK Version 3.9 - Improve Last/Title parser to handle names with spaces
// SDK Version 4.0 - Add RealID field to PDF, update Mag for MN/CO/FL, update CyUSB.dll for M280 imaging
// SDK Version 4.1 - Improve barcode parsing
// SDK Version 4.3.5 - Update parsing license to QLM - license file based on computer Name.
namespace IdParserM280Works
{
    public partial class frmMain : Form
    {
        #region Local Variables
        public static SerialPort comport = new SerialPort();
        MagQLibNet.Magstripe mc;
        PdfQLibNet.Barcode bc;
        bool showActivation; //used to control the use of the Activation form, embeded in the parsers



        byte[] MagBuff = new byte[315]; //Magstrip reader buffer
        byte[] buff = new byte[2000]; // PDF417 barcode reader buffer 
        byte[] Rawbuff = new byte[2000]; // RawData buffer used for data logging

        string displaystring;

        int ss1 = 26; //start sentinel 1
        int ss2 = 27; //start sentinel 2
        int ss3 = 28; //start sentinel 3
        int MBD = 29; //Magstripe Barcode Designator

        string rawdata; //holds hex encoded raw data 
        string ASCIIdata; // holds eyeball encoded data
        DateTime d1 = DateTime.Now;

// M280 Support
        USBDeviceList usbDevices;
        CyUSBDevice MyDevice;
        CyBulkEndPoint inEndpoint = null;
        CyControlEndPoint ep0 = null;
        int BufNum = 0;
        int BufSz;
        int QueueSz;
        int Successes;
        int Failures;
        int TotalPixSize;
        double XferBytes;
        bool bRunning;
        bool bInitM280;
        bool bSysBusy;
        byte[][] DataBuf;
        byte[] M280Status = new byte[M280DEF.STATUS_SIZE];

        // Image Info
        int DatumPos_sX = 0, DatumPos_sY = 0, CropSize_X = 0, CropSize_Y = 0, AngleInfo = 0;

        Bitmap CroppedImage;

        // Start capture Thread
        Thread tStartCap;

        // These are needed to close the app from the Thread exception(exception handling)
        delegate void ExceptionCallback();
        ExceptionCallback handleException;

        // These are  needed for Thread to make image view function
        delegate void Image_View();
        Image_View Img_View;

        //scanner safety
        bool ScanerBusy = false;

        // WIN10 UPDATE (VER:1803) -->
        static bool bRelease = true;


        #endregion


        public frmMain()
        {
            InitializeComponent();

            // When data is recieved through the port, call this method
            comport.DataReceived += new SerialDataReceivedEventHandler(port_DataReceived);

            //Initialize the time stamp
            ReaderBuffMgr.GetTimeStamp();

            //Enable the activation form for the licensed parsing DLLs
            showActivation = true; //When false, and no license file, a log file:IDParser_EventLog.txt is created, that contains the ComputerID.

            //M280 Initialization
            // For USB Camera
            // Create the list of USB devices attached to the CyUSB.sys driver.
            usbDevices = new USBDeviceList(CyConst.DEVICES_CYUSB);

            // Setup the callback routine for NullReference exception handling
            handleException = new ExceptionCallback(ThreadException);

            // Setup the callback routine for Image view from M280 board
            Img_View = new Image_View(ImageView);

            //Assign event handlers for device attachment and device removal.
            usbDevices.DeviceAttached += new EventHandler(usbDevices_DeviceAttached);
            usbDevices.DeviceRemoved += new EventHandler(usbDevices_DeviceRemoved);
            // Detect power event
            SystemEvents.PowerModeChanged += OnPowerChange; 

            //Set and search the device with VID-PID 04b4-1003 and if found, selects the end point
            SetDevice();

            //Initialize value
            if (M280DEF.PixFormat == PixelFormat.Format16bppRgb565)
            {
                TotalPixSize = M280DEF.Image_Xsize * M280DEF.Image_Ysize * 2;
            }

            if ((TotalPixSize % (512 * M280DEF.Packet_Xfer)) != 0)
            {
                MessageBox.Show("Please check pixel size!");
            }

            this.BufNum = TotalPixSize / (512 * M280DEF.Packet_Xfer);
            this.timCheckStatus.Tick += new EventHandler(CheckState_Tick);
            this.timCheckStatus.Interval = M280DEF.StatusGetInterval;
            this.bInitM280 = false;
            this.bSysBusy = false;


        }

        private void OnPowerChange(object s, PowerModeChangedEventArgs e)
        {
            switch (e.Mode)
            {
                case PowerModes.Resume: Application.Restart(); //restart the app if the PC goes into low power, causing the scanner to be busy.
                    break;
                case PowerModes.Suspend: break;
            }
        }


        private void SetDevice()
        {
            MyDevice = usbDevices[M280DEF.USB_VID, M280DEF.USB_PID] as CyUSBDevice;

            if (MyDevice != null)
            {
                // USB High Speed Check
                /*
                if (MyDevice.bHighSpeed == false)
                {
                    MessageBox.Show("M280 supports USB2.0 only.");
                    MyDevice = null;
                    inEndpoint = null;
                    ep0 = null;
                } 
                else
                */
                if (inEndpoint == null)
                {
                    // Set the IN and OUT endpoints           
                    inEndpoint = MyDevice.EndPointOf(0x86) as CyBulkEndPoint;
                    ep0 = MyDevice.ControlEndPt;
                    // Set time out
                    inEndpoint.TimeOut = M280DEF.TRANSFER_TIMEOUT;
                    ep0.TimeOut = M280DEF.TRANSFER_TIMEOUT;
                    // Get Image Info 
                    GetImageInfo();
                    // Initialize Camera module
                    SendCommand(M280DEF.CMD_INIT_CAMERA, 0, 0);
                    // Turn on USB connected LED
                    labUSBConImg.Image = global::IdParserM280Works.Properties.Resources.green3;
                    // Start Check status
                    this.timCheckStatus.Start();
                }
            }
            else
            {
                inEndpoint = null;
                ep0 = null;

                // Turn off USB connected LED
                labUSBConImg.Image = global::IdParserM280Works.Properties.Resources.gray3;
                labSysReadyImg.Image = global::IdParserM280Works.Properties.Resources.gray3;
                labReadyImg.Image = global::IdParserM280Works.Properties.Resources.gray3;
                labBusyImg.Image = global::IdParserM280Works.Properties.Resources.gray3;
                labButtonImg.Image = global::IdParserM280Works.Properties.Resources.gray3;

                // Stop checking status thread
                this.timCheckStatus.Stop();
                this.bInitM280 = false;
                this.bSysBusy = false;
            }
        }


        #region M280 image
        void usbDevices_DeviceRemoved(object sender, EventArgs e)
        {
            MyDevice = null;
            inEndpoint = null;
            SetDevice();
        }

        //          This is the event handler for device attachment. This method  searches for the device with VID-PID 04b4-1003

        void usbDevices_DeviceAttached(object sender, EventArgs e)
        {
            SetDevice();
        }

        private void btnScan_Click(object sender, EventArgs e)
        {
            if (ScanerBusy == true) return;

            try
            {

                if (MyDevice == null || ep0 == null || inEndpoint == null)
                {
                    MessageBox.Show("Please Connect M280 Scanner to PC");
                    return;
                }
                if (bRunning == true || this.bInitM280 == false || this.bSysBusy == true)
                {
                    MessageBox.Show("Scanner Busy... Try again.");
                    return;


                }
                // WIN10 UPDATE (VER:1803) -->
                if (bRelease == false) return;
                bRelease = false;
                bRunning = true;

                BufSz = inEndpoint.MaxPktSize * M280DEF.Packet_Xfer;
                QueueSz = BufNum;

                inEndpoint.XferSize = BufSz;

                // Initializing Data Buffer
                DataBuf = new byte[TotalPixSize / BufSz][];
                for (int i = 0; i < (TotalPixSize / BufSz); i++)
                {
                    DataBuf[i] = new byte[BufSz];
                }

                tStartCap = new Thread(new ThreadStart(XferThread));
                tStartCap.IsBackground = true;
                tStartCap.Priority = ThreadPriority.Highest;
                tStartCap.Start();
                // Send Start capture command to M280
                SendCommand(M280DEF.CMD_ST_CAP, 0, 0);

                //make a noise
                SystemSounds.Beep.Play();
                ScanerBusy = true; // cleared in XferData
            }
            catch
            {
                MessageBox.Show("Check Card Scanner to PC connection");
            }

        }
        public unsafe void XferThread()
        {
            // Setup the queue buffers
            byte[][] cmdBufs = new byte[QueueSz][];
            byte[][] xferBufs = new byte[QueueSz][];
            byte[][] ovLaps = new byte[QueueSz][];
            ISO_PKT_INFO[][] pktsInfo = new ISO_PKT_INFO[QueueSz][];
            int xStart = 0;

            try
            {
                LockNLoad(ref xStart, cmdBufs, xferBufs, ovLaps, pktsInfo);
            }
            catch (NullReferenceException e)
            {
                // This exception gets thrown if the device is unplugged 
                // while we're streaming data
                e.GetBaseException();
                this.Invoke(handleException);
            }
        }

        public unsafe void LockNLoad(ref int j, byte[][] cBufs, byte[][] xBufs, byte[][] oLaps, ISO_PKT_INFO[][] pktsInfo)
        {
            // Allocate one set of buffers for the queue
            cBufs[j] = new byte[CyConst.SINGLE_XFER_LEN];
            xBufs[j] = new byte[BufSz];
            oLaps[j] = new byte[20];
            //pktsInfo[j] = new ISO_PKT_INFO[PPX];
            pktsInfo[j] = new ISO_PKT_INFO[M280DEF.Packet_Xfer];

            fixed (byte* tL0 = oLaps[j], tc0 = cBufs[j], tb0 = xBufs[j])  // Pin the buffers in memory
            {
                OVERLAPPED* ovLapStatus = (OVERLAPPED*)tL0;
                ovLapStatus->hEvent = (IntPtr)PInvoke.CreateEvent(0, 0, 0, 0);

                // Pre-load the queue with a request
                int len = BufSz;
                inEndpoint.BeginDataXfer(ref cBufs[j], ref xBufs[j], ref len, ref oLaps[j]);

                j++;

                if (j < QueueSz)
                    LockNLoad(ref j, cBufs, xBufs, oLaps, pktsInfo);  // Recursive call to pin next buffers in memory
                else
                {
                    XferData(cBufs, xBufs, oLaps, pktsInfo);          // All loaded. Let's go!
                }
            }
        }

        public unsafe void XferData(byte[][] cBufs, byte[][] xBufs, byte[][] oLaps, ISO_PKT_INFO[][] pktsInfo)
        {
            try
            {
                int k = 0;
                int len = 0;
                int pDataBF = 0;

                Successes = 0;
                Failures = 0;
                XferBytes = 0;

                //            for (; bRunning; )
                while (bRunning)
                {
                    // WaitForXfer
                    fixed (byte* tmpOvlap = oLaps[k])
                    {
                        OVERLAPPED* ovLapStatus = (OVERLAPPED*)tmpOvlap;
                        if (!inEndpoint.WaitForXfer(ovLapStatus->hEvent, 500))
                        {
                            inEndpoint.Abort();
                            PInvoke.WaitForSingleObject(ovLapStatus->hEvent, 500);
                        }
                    }

                    // FinishDataXfer
                    if (inEndpoint.FinishDataXfer(ref cBufs[k], ref xBufs[k], ref len, ref oLaps[k]))
                    {
                        XferBytes += len;
                        Successes++;
                        Array.Copy(xBufs[k], 0, DataBuf[pDataBF], 0, len);
                        pDataBF++;
                    }
                    else
                    {
                        Failures++;
                    }
                    // WIN10 UPDATE (VER:1803) -->
                    // Re-submit this buffer into the queue
                    len = BufSz;
                    inEndpoint.BeginDataXfer(ref cBufs[k], ref xBufs[k], ref len, ref oLaps[k]);

                    k++;
                    if (k == QueueSz)  // Finish
                    {
                        k = 0;
                        Thread.Sleep(1);
                        if (Failures == 0)
                        {
                            this.Invoke(Img_View);
                        }
                        else
                        {
                            // Scanner busy
                            MessageBox.Show("Scanner Busy! Please Try again.");
                            // Vacate the trash.
                            len = 1024 * 4;
                            byte[] buf = new byte[len];
                            inEndpoint.Abort();
                            inEndpoint.Reset();
                            inEndpoint.XferData(ref buf, ref len);
                            Thread.Sleep(100);
                        }
                        bRunning = false;
                        ScanerBusy = false; // set in btnScan_Click
                    }
                }
                // WIN10 UPDATE (VER:1803) -->
                inEndpoint.Abort();
                bRelease = true;
            }
            catch
            {
                MessageBox.Show("Scanner Busy! Please Try again.");
            }
        }

        public bool SendCommand(byte Cmd, ushort val, ushort index)
        {
            bool ret = false;
            if (MyDevice == null || ep0 == null) return ret;

            ep0.Target = CyConst.TGT_DEVICE;
            ep0.ReqType = CyConst.REQ_VENDOR;
            ep0.Direction = CyConst.DIR_TO_DEVICE;
            ep0.ReqCode = Cmd;
            ep0.Value = val;
            ep0.Index = index;
            int len = 0;
            byte[] buf = new byte[1];
            ret = ep0.XferData(ref buf, ref len);
            return ret;
        }

        public bool ReadCommand(byte Cmd, ref byte[] Read, ref int Len, ushort val, ushort index)
        {
            bool ret = false;
            if (MyDevice == null || ep0 == null) return ret;
            ep0.Target = CyConst.TGT_DEVICE;
            ep0.ReqType = CyConst.REQ_VENDOR;
            ep0.Value = val;
            ep0.Index = index;
            ep0.ReqCode = Cmd;
            ret = ep0.Read(ref Read, ref Len);
            return ret;
        }

        private void CheckState_Tick(object sender, EventArgs e)
        {
            this.timCheckStatus.Stop();

            int len = M280DEF.STATUS_SIZE;
            UInt16 StatusValue;

            if (MyDevice == null || ep0 == null) return;

            this.ReadCommand(M280DEF.CMD_GET_STATE, ref this.M280Status, ref len, 0, 0);

            // Parsing
            StatusValue = BitConverter.ToUInt16(M280Status, 0);

            // Check Push button
            if ((StatusValue & M280DEF.stat_CapDet) == M280DEF.stat_CapDet)
            {
                labButtonImg.Image = global::IdParserM280Works.Properties.Resources.green3;
                this.btnScan_Click(0, null);
            }
            else
            {
                labButtonImg.Image = global::IdParserM280Works.Properties.Resources.gray3;
            }

            // Check Busy LED
            if ((StatusValue & M280DEF.stat_BusyLED) == M280DEF.stat_BusyLED)
            {
                labBusyImg.Image = global::IdParserM280Works.Properties.Resources.red3;
            }
            else
            {
                labBusyImg.Image = global::IdParserM280Works.Properties.Resources.gray3;
            }

            // Check Ready LED
            if ((StatusValue & M280DEF.stat_ReadyLED) == M280DEF.stat_ReadyLED)
            {
                labReadyImg.Image = global::IdParserM280Works.Properties.Resources.green3;
            }
            else
            {
                labReadyImg.Image = global::IdParserM280Works.Properties.Resources.gray3;
            }

            // Check System Ready
            if ((StatusValue & M280DEF.stat_CamInit) == M280DEF.stat_CamInit)
            {
                labSysReadyImg.Image = global::IdParserM280Works.Properties.Resources.green3;
                this.bInitM280 = true;
            }
            else
            {
                labSysReadyImg.Image = global::IdParserM280Works.Properties.Resources.gray3;
                this.bInitM280 = false;
            }

            // Check System Busy
            if ((StatusValue & M280DEF.stat_SysBusy) == M280DEF.stat_SysBusy)
            {
                labSysBusyImg.Image = global::IdParserM280Works.Properties.Resources.green3;
                this.bSysBusy = true;
            }
            else
            {
                labSysBusyImg.Image = global::IdParserM280Works.Properties.Resources.gray3;
                this.bSysBusy = false;
            }

            this.timCheckStatus.Start();
        }


        public void GetImageInfo()
        {
            if (MyDevice == null || ep0 == null) return;
            int len = M280DEF.IMGINFO_SIZE;
            byte[] dta = new byte[len];

            if (this.ReadCommand(M280DEF.CMD_GET_IMGINFO, ref dta, ref len, 0, 0) == true)
            {
                // Convert Endian
                Array.Reverse(dta, 0, 2);   // DatumPos_sX
                Array.Reverse(dta, 2, 2);   // DatumPos_sY
                Array.Reverse(dta, 4, 2);   // CropSize_X
                Array.Reverse(dta, 6, 2);   // CropSize_Y
                Array.Reverse(dta, 14, 2);  // DPI

                this.DatumPos_sX = BitConverter.ToUInt16(dta, 0);
                this.DatumPos_sY = BitConverter.ToUInt16(dta, 2);
                this.CropSize_X = BitConverter.ToUInt16(dta, 4);
                this.CropSize_Y = BitConverter.ToUInt16(dta, 6);
                this.AngleInfo = (int)dta[13];
            }
            else
            {
                // Cann't find Image Info
                MessageBox.Show("Can not find Image Info Please check Firmware version!");
                this.DatumPos_sX = (M280DEF.Image_Xsize - M280IMGDEF.IMG_CARDSZ_X) / 2;
                this.DatumPos_sY = (M280DEF.Image_Ysize - M280IMGDEF.IMG_CARDSZ_Y) / 2;
                this.CropSize_X = M280IMGDEF.IMG_CARDSZ_X;
                this.CropSize_Y = M280IMGDEF.IMG_CARDSZ_Y;
                this.AngleInfo = (int)M280IMGDEF.IMG_ANGLE;
            }
        }

        public void ThreadException()
        {
            bRunning = false;
            tStartCap = null;
            this.timCheckStatus.Dispose();
        }

        public void ImageView()
        {
            Bitmap SourceImage = CreateBitmap(M280DEF.Image_Xsize, M280DEF.Image_Ysize, M280DEF.PixFormat);

            // Adjust Angle
            if (chkTiltAdj.Checked == true)
            {
                SourceImage = Utilities.RotateImage(SourceImage, new PointF((float)(this.DatumPos_sX + this.CropSize_X / 2),
                                            (float)(this.DatumPos_sY + this.CropSize_Y / 2)), (byte)this.AngleInfo);
            }

            // Crop Image
            Rectangle CropRect = new Rectangle(this.DatumPos_sX, this.DatumPos_sY,
                                                this.CropSize_X, this.CropSize_Y);
            this.CroppedImage = SourceImage.Clone(CropRect, M280DEF.PixFormat);

            SourceImage.Dispose();
            int width = picImage.Width;
            int height = picImage.Height;
            Bitmap small = new Bitmap(this.CroppedImage, width, height);
            picImage.Image = small;

        }

        public Bitmap CreateBitmap(int Width, int Height, PixelFormat Pfmt)
        {
            if (Pfmt == PixelFormat.Format24bppRgb)
            {
                if ((Width * Height * 3) != (TotalPixSize))
                    return null;
            }
            else if (Pfmt == PixelFormat.Format16bppRgb565)
            {
                if ((Width * Height * 2) != (TotalPixSize))
                    return null;
            }
            else
            {
                return null;
            }

            try
            {
                Bitmap Canvas = new Bitmap(Width, Height, Pfmt);
                BitmapData CanvasData = Canvas.LockBits(new Rectangle(0, 0, Width, Height),
                                    ImageLockMode.WriteOnly, Pfmt);

                unsafe
                {
                    byte* Ptr = (byte*)CanvasData.Scan0.ToPointer();

                    for (int Y = 0; Y < TotalPixSize / BufSz; Y++)
                    {

                        for (int X = 0; X < BufSz; X++)
                        {
                            // Swap Data
                            if (X % 2 == 1)
                            {
                                *Ptr = DataBuf[Y][X];
                                Ptr++;
                                *Ptr = DataBuf[Y][X - 1];
                                Ptr++;
                            }
                        }
                    }
                }
                Canvas.UnlockBits(CanvasData);
                return Canvas;

            }
            catch (Exception)
            {
                return null;
            }

        }
        #endregion


        #region Local Functions
        /// <summary> Convert a string of hex digits (ex: E4 CA B2) to a byte array. </summary>
        /// <param name="s"> The string containing the hex digits (with or without spaces). </param>
        /// <returns> Returns an array of bytes. </returns>
        public static byte[] HexStringToByteArray(string s)
        {
            s = s.Replace(" ", "");
            byte[] buffer = new byte[s.Length / 2];
            for (int i = 0; i < s.Length; i += 2)
                buffer[i / 2] = (byte)Convert.ToByte(s.Substring(i, 2), 16);
            return buffer;
        }

        /// <summary> Converts an array of bytes into a formatted string of hex digits (ex: E4 CA B2)</summary>
        /// <param name="data"> The array of bytes to be translated into a string of hex digits. </param>
        /// <returns> Returns a well formatted string of hex digits with spacing. </returns>
        public string ByteArrayToHexString(byte[] data)
        {
            StringBuilder sb = new StringBuilder(data.Length * 3);
            foreach (byte b in data)
                sb.Append(Convert.ToString(b, 16).PadLeft(2, '0').PadRight(3, ' '));
            return sb.ToString().ToUpper();
        }

        #endregion



        #region Events
        private void port_DataReceived(object sender, SerialDataReceivedEventArgs e)
        {

            // Obtain the number of bytes waiting in the port's buffer
            int bytes = comport.BytesToRead;

            // Create a byte array buffer to hold the incoming data
            byte[] buffer = new byte[bytes];

            // Read the data from the port and store it in our buffer
            comport.Read(buffer, 0, bytes);
            ReaderBuffMgr.DataBuilder(buffer, Environment.TickCount);
            d1 = DateTime.Now; //update timestamp

                //test for a complete scan before calling pdf417 parsing
                if ((ReaderBuffMgr.RawBytes[ss1] == 0x01) && (ReaderBuffMgr.RawBytes[ss2] == 0x02) && (ReaderBuffMgr.RawBytes[ss3] == 0x5D) && (ReaderBuffMgr.RawBytes[MBD] == 0x4C) && (ReaderBuffMgr.RawBytes[ReaderBuffMgr.PrevLng - 1] == 0x04))
                {
                    Array.Resize(ref buff, ReaderBuffMgr.PrevLng); //You MUST resize the global to pass only current data
                    ReaderBuffMgr.RawBytes[ReaderBuffMgr.PrevLng] = 0;//insert null to create printable string
                    Array.Copy(ReaderBuffMgr.RawBytes, 0, buff, 0, ReaderBuffMgr.PrevLng);

                    ReaderBuffMgr.PrevLng = 0; //reset conters
                    ReaderBuffMgr.count = 0;  //for next scan

                    //the following code supports data logging to a file
                    Array.Resize(ref Rawbuff, buff.Length-ss1);
                    Array.Copy(buff, ss1, Rawbuff, 0, buff.Length-ss1);
                    rawdata = ByteArrayToHexString(Rawbuff); //convert binary to hex string, 
                    FileStore.WriteString2File(rawdata); //write raw data to file
                    ASCIIdata = new ASCIIEncoding().GetString(Rawbuff);
                    ASCIIdata = string.Concat(ASCIIdata, d1.ToString());
                    FileStore.WriteString2File(ASCIIdata); //store the ascii data and time stamp
                    //end of data logging



                    try
                    {
                        this.Invoke(new EventHandler(SendPDF));
                    }
                    catch
                    {
                        MessageBox.Show("Invalid Barcode");
                    }
                    displaystring = PdfQLibNet.Barcode.PDF_String;
                    this.Invoke(new EventHandler(SetText));
                }

                //test for magstripe read from scanner
                if ((ReaderBuffMgr.RawBytes[ss1] == 01) && (ReaderBuffMgr.RawBytes[ss2] == 02) && (ReaderBuffMgr.RawBytes[ss3] == 0x5D) && (ReaderBuffMgr.RawBytes[MBD] == 0x4D) && (ReaderBuffMgr.RawBytes[ReaderBuffMgr.PrevLng - 1] == 0x04))
                {
                    Array.Resize(ref MagBuff, ReaderBuffMgr.PrevLng); //You MUST resize the global to pass only current data
                    ReaderBuffMgr.RawBytes[ReaderBuffMgr.PrevLng] = 0;//insert null to create printable string
                    Array.Copy(ReaderBuffMgr.RawBytes, 0, MagBuff, 0, ReaderBuffMgr.PrevLng);

                    ReaderBuffMgr.PrevLng = 0; //reset counters
                    ReaderBuffMgr.count = 0; //for next swipe

                    //the following code supports data logging to a file
                    Array.Resize(ref Rawbuff, MagBuff.Length-ss1);
                    Array.Copy(MagBuff, ss1, Rawbuff, 0, MagBuff.Length - ss1);
                    rawdata = ByteArrayToHexString(Rawbuff); //convert binary to hex string
                    FileStore.WriteString2File(rawdata); //write it to file
                    ASCIIdata = new ASCIIEncoding().GetString(Rawbuff);
                    ASCIIdata = string.Concat(ASCIIdata, d1.ToString());
                    FileStore.WriteString2File(ASCIIdata); //store the ascii data and time stamp
                    //end of data logging


                    try
                    {
                        this.Invoke(new EventHandler(SendMAG));
                    }
                    catch
                    {
                        MessageBox.Show("Invalid Magstripe");
                    }
                    displaystring = MagQLibNet.Magstripe.track1 + MagQLibNet.Magstripe.track2 + MagQLibNet.Magstripe.track3;
                    this.Invoke(new EventHandler(SetText));
                }

                //this will catch bad barcode reads
                if ((ReaderBuffMgr.RawBytes[ss1] == 01) && (ReaderBuffMgr.RawBytes[ss2] == 02) && (ReaderBuffMgr.RawBytes[ss3] == 0x4E) && (ReaderBuffMgr.RawBytes[MBD] == 0x52) && (ReaderBuffMgr.RawBytes[ReaderBuffMgr.PrevLng - 1] == 0x04))
                {
                    ReaderBuffMgr.PrevLng = 0;
                    ReaderBuffMgr.count = 0;

                    displaystring = "\r\n Bad Barcode Read, Try Again";
                    this.Invoke(new EventHandler(SetText));

                }

                //this catches the ACK from the Ping command
                if ((ReaderBuffMgr.RawBytes[0] == 0x06) && (ReaderBuffMgr.PrevLng == 0x01))
                {
                    ReaderBuffMgr.PrevLng = 0;
                    ReaderBuffMgr.count = 0;
                    // Show the user the incoming data in hex format
                    displaystring = "ACK!";
                    this.Invoke(new EventHandler(SetText));
                }
        }


        private void SetText(object o, EventArgs e)
        {
            txtOutput.Text = displaystring;
        }


        private void SendMAG(object sender, EventArgs e)
        {
            int ret = 0;
            try
            {
                ret = mc.ParseDL(MagBuff); // Parse the buffer
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "Magstripe Parsing Exception");
                return;
            }


           // ret = Magstripe.ParseDL(MagBuff);

                lblVersion.Text = Magstripe.Version;
                txtTitle.Text = Magstripe.titleP_DL;
                txtFirst.Text = Magstripe.firstnameP_DL;
                txtMid.Text = Magstripe.midnameP_DL;
                txtLast.Text = Magstripe.lastnameP_DL;
                txtAddress.Text = Magstripe.addrP_DL;
                txtCity.Text = Magstripe.cityP_DL;
                txtState.Text = Magstripe.stateP_DL;
                txtZip.Text = Magstripe.zipP_DL;
                txtID.Text = Magstripe.dlP_DL;
                txtEXP.Text = Magstripe.expP_DL;
                txtDobMMDD.Text = Magstripe.dobMMDDP_DL;
                txtDobYear.Text = Magstripe.dobyearP_DL;
                txtHeight.Text = Magstripe.HeightP_DL;
                txtEye.Text = Magstripe.EyeP_DL;
                txtSEX.Text = Magstripe.SexP_DL;
                txtWeight.Text = Magstripe.WeightP_DL;
                txtClass.Text = Magstripe.ClassP_DL;
                txtRestriction.Text = Magstripe.RestrictionsP_DL;
                txtEndorsement.Text = Magstripe.EndorsementsP_DL;
                txtHair.Text = Magstripe.HairP_DL;
               
                txtScannerSN.Text = Magstripe.ScannerSN;
                txtAddr2.Text = string.Empty; // not available on magnetic
                txtCountry.Text = string.Empty; //n/a
                txtRace.Text = string.Empty; // n/a
                txtDBD.Text = string.Empty; // n/a
                txtRealiD.Text = string.Empty; // n/a

                txtFirstCC.Text = Magstripe.firstname_CC;
                txtLastCC.Text = Magstripe.lastname_CC;
                txtExpMMCC.Text = Magstripe.EXPMM_CC;
                txtEXPYYCC.Text = Magstripe.EXPYY_CC;
                txtPANCC.Text = Magstripe.PAN_CC;


        }

        private void SendPDF(object sender, EventArgs e)
        {
            int i = -1;
            try
            {
                i = bc.Parse417(buff); // Parse the buffer
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "Barcode Parsing Exception");
                return;
            }


            //i = Barcode.Parse417(buff);
            lblVersion.Text = Barcode.Version;
            //Loading Gui Object with Parsed Fields
            txtTitle.Text = Barcode.Title_DL;
            txtFirst.Text = Barcode.First_DL;
            txtMid.Text = Barcode.Mid_DL;
            txtLast.Text = Barcode.Last_DL;
            txtAddress.Text = Barcode.Addr_DL;
            txtAddr2.Text = Barcode.Addr2_DL;
            txtCity.Text = Barcode.City_DL;
            txtState.Text = Barcode.State_DL;
            txtZip.Text = Barcode.Zip_DL;
            txtID.Text = Barcode.Dl_DL;
            txtSEX.Text = Barcode.Sex_DL;
            txtEXP.Text = Barcode.Exp_DL;
            txtDobMMDD.Text = Barcode.DobMMDD_DL;
            txtDobYear.Text = Barcode.DobYear_DL;

            txtEye.Text = Barcode.Eye_DL;
            txtHeight.Text = Barcode.Height_DL;
            txtClass.Text = Barcode.Class_DL;
            txtRestriction.Text = Barcode.Rest_DL;
            txtEndorsement.Text = Barcode.Endr_DL;
            txtWeight.Text = Barcode.Wt_DL;
            txtHair.Text = Barcode.Hair_DL;
            txtCountry.Text = Barcode.Country_DL;
            txtRace.Text = Barcode.Race_DL;
            txtDBD.Text = Barcode.DocIssDate_DL;
            txtScannerSN.Text = Barcode.ScannerSN;
            txtRealiD.Text = Barcode.RealID_DL;



        }

        private void frmMain_Load(object sender, EventArgs e) //load the libraries here to improve response time.
        {
            try
            {
                string KeyStatus;
                bc = new PdfQLibNet.Barcode(showActivation); //use the constructor with bool to display activation form, when no license is found
                KeyStatus = "KeyResponse = " + Barcode.KeyResponse.ToString() + "\r\n";
                KeyStatus += "KeyStatus = " + Barcode.KeyStatus + "\r\n";
                KeyStatus += "S.W.MaintRenewal = " + Barcode.KeySwMaintRenewDate + "\r\n";
                KeyStatus += "ComputerID = " + Barcode.KeyComputerID + "\r\n";

                displaystring = KeyStatus;
                this.Invoke(new EventHandler(SetText));
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "Error loading PdfQLibNet");
            }


            try
            {

                mc = new MagQLibNet.Magstripe(showActivation); // uses same license file, same Key properties

            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "Error loading MagQLibNet");
            }


        }

        private void btnOpenPort_Click(object sender, EventArgs e)
        {
            // If the port is open, close it.
            if (comport.IsOpen) comport.Close();
            else
            {
                comport.BaudRate = Settings.Default.BaudRate;
                comport.DataBits = Settings.Default.DataBits;
                comport.StopBits = Settings.Default.StopBits;
                comport.Parity = Settings.Default.Parity;
                comport.PortName = Settings.Default.PortName;
                // Open the port
                try{
                comport.Open();
                    }
            catch (Exception px)
            {
                MessageBox.Show(px.Message,"Port already in use");
            }


            }
            if (comport.IsOpen)
            {
                btnOpenPort.Text = "&Close Port";
            }
            else btnOpenPort.Text = "&Open Port";

        }

        private void btnPort_Click(object sender, EventArgs e)
        {
            (new frmSettings()).ShowDialog();
        }


        private void btnTest2_Click(object sender, EventArgs e)
        {
            if (comport.IsOpen)
            {
                //this commands pings the port to see if the reader is connected
                //'01'<SERIEN=0><REGIEN=0>'04'
                byte[] data1 = HexStringToByteArray("01 3C 54 58 50 49 4E 47 3A 3E 04"); //
                // Send the binary data out the port
                try
                {
                    comport.Write(data1, 0, data1.Length);
                }
                catch (Exception px)
                {
                    MessageBox.Show(px.Message, "Port not available");
                }
            }

        }
       

        private void btnProgram_Click(object sender, EventArgs e)
        {
            if (comport.IsOpen)
            {
                ////this sets every parameter for reader to work with TokenWorks IDParser Library. Scanner must also be registered to work with IDParser.
                byte[] data1 = HexStringToByteArray("01 3C 53 45 52 49 45 4E 3D 31 3E 3C 52 45 47 49 45 4E 3D 31 3E 3C 4E 4F 52 45 41 44 3D 31 3E 3C 50 52 45 43 49 44 3D 31 3E 3C 50 52 45 46 49 58 3D 5B 53 4F 48 5D 5B 53 54 58 5D 3E 3C 53 55 46 46 49 58 3D 5B 45 54 58 5D 5B 45 4F 54 5D 3E 3C 4D 41 47 31 45 4E 3D 31 3E 3C 4D 41 47 32 45 4E 3D 31 3E 3C 4D 41 47 33 45 4E 3D 31 3E 3C 50 44 46 34 31 37 3D 31 3E 3C 4D 41 47 52 41 57 3D 30 3E 3C 52 45 47 4B 45 59 3A 3E 3C 53 45 52 49 41 4C 3A 3E 04"); //programming sequence for Scanner

                // Send the binary data out the port
                comport.Write(data1, 0, data1.Length);
            }

        }

        #endregion

 


  
    }
}